import { ref, onMounted, onUnmounted } from 'vue';

export function useIdleTimer(timeout = 900000, promptTimeout = 60000) { // Default: 15min timeout, 1min prompt
  const isIdle = ref(false);
  const isPrompted = ref(false);

  let idleTimer = null;
  let promptTimer = null;

  const events = ['mousemove', 'mousedown', 'keydown', 'touchstart', 'scroll'];

  const reset = () => {
    isIdle.value = false;
    isPrompted.value = false;

    clearTimeout(idleTimer);
    clearTimeout(promptTimer);

    idleTimer = setTimeout(() => {
        isIdle.value = true;
        isPrompted.value = true;
        
        // Start a countdown to logout if user doesn't respond to prompt
        promptTimer = setTimeout(() => {
            // This is where the final logout action will be triggered from the component
        }, promptTimeout);

    }, timeout - promptTimeout);
  };

  onMounted(() => {
    events.forEach(event => window.addEventListener(event, reset));
    reset();
  });

  onUnmounted(() => {
    events.forEach(event => window.removeEventListener(event, reset));
    clearTimeout(idleTimer);
    clearTimeout(promptTimer);
  });

  return {
    isIdle,
    isPrompted,
    reset,
  };
} 