import axios from 'axios';
import router from '../Router';
import { useAuthStore } from "@/Stores/Auth.js";

const csrfToken = document.head.querySelector('meta[name="csrf-token"]');

const http_axios = axios.create({
    baseURL: (import.meta.env.VITE_BASE_URL || '/') + 'api/',
    headers: {
        'X-CSRF-TOKEN': csrfToken.content,
        "Content-type": "application/json",
        "Access-Control-Allow-Origin": "*",
    },
});

http_axios.interceptors.request.use((request) => {
    const token = localStorage.getItem('token');
    console.log('Axios interceptor token:', token, 'for URL:', request.url);
    if (token) {
        request.headers.Authorization = 'Bearer ' + token;
    }
    return request;
});

http_axios.interceptors.response.use(
    response => response,
    async error => {
        const originalRequest = error.config;
        const userStore = useAuthStore();
        const token = userStore.getToken();

        if (
            error.response &&
            error.response.status === 401 &&
            !originalRequest._retry &&
            !originalRequest.url.endsWith('/auth/login') &&
            !originalRequest.url.endsWith('/auth/refresh') &&
            token
        ) {
            // Check if it's an automatic logout due to inactivity
            const message = error.response.data?.message || '';
            if (message.includes('inactivity') || message.includes('expired')) {
                // Clear all stored data
                localStorage.removeItem('token');
                localStorage.removeItem('refresh_token');
                localStorage.removeItem('user');
                
                // Show notification to user
                if (typeof window !== 'undefined' && window.alert) {
                    alert('You have been automatically logged out due to inactivity. Please login again.');
                }
                
                // Redirect to login page
                if (typeof window !== 'undefined' && window.location) {
                    window.location.href = '/login';
                }
                return Promise.reject(error);
            }

            originalRequest._retry = true;
            const currentRefreshToken = userStore.getRefreshToken();
            if (!currentRefreshToken) {
                userStore.logout();
                router.push('/');
                return Promise.reject(error);
            }
            try {
                const refreshResponse = await http_axios.post('auth/refresh', {
                    refresh_token: currentRefreshToken
                });
                const newAccessToken = refreshResponse.data.access_token;
                const newRefreshToken = refreshResponse.data.refresh_token;
                userStore.setToken(newAccessToken);
                userStore.setRefreshToken(newRefreshToken);
                originalRequest.headers['Authorization'] = 'Bearer ' + newAccessToken;
                return http_axios(originalRequest);
            } catch (refreshError) {
                userStore.logout();
                router.push('/');
            }
        } else if (error.response && [401, 403].includes(error.response.status)) {
            userStore.logout();
            router.push('/');
        }
        return Promise.reject(error);
    }
);

export default http_axios;
